import React, { useEffect, useState } from 'react';
import { Card, CardHeader, CardTitle, CardContent } from '../components/ui/card';
import { Button } from '../components/ui/button';
import { useAuth } from '../hooks/use-auth';
import { useTranslation } from 'react-i18next';

function useIsMobile(breakpoint = 768) {
  const [isMobile, setIsMobile] = useState(false);
  useEffect(() => {
    const check = () => setIsMobile(window.matchMedia(`(max-width: ${breakpoint}px)`).matches);
    check();
    window.addEventListener('resize', check);
    return () => window.removeEventListener('resize', check);
  }, [breakpoint]);
  return isMobile;
}

const RoleManagement: React.FC = () => {
  const { t, i18n } = useTranslation();
  const { userPermissions } = useAuth();
  const [roles, setRoles] = useState<any[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState<string | null>(null);
  const isRTL = i18n.dir() === 'rtl';
  const isMobile = useIsMobile();

  const PERMISSIONS = [
    { key: 'can_open_valve', label: t('roleManagement.openValve', 'Open Valve') },
    { key: 'can_close_valve', label: t('roleManagement.closeValve', 'Close Valve') },
    { key: 'can_change_maximum', label: t('roleManagement.changeMaximum', 'Change Maximum') },
    { key: 'can_approve_maximum', label: t('roleManagement.approveMaximum', 'Approve Maximum') },
    { key: 'can_change_env', label: t('roleManagement.changeEnv', 'Change Environment Variables') },
    { key: 'can_change_ui', label: t('roleManagement.changeUI', 'Change UI Configuration') },
    { key: 'can_manage_users', label: t('roleManagement.manageUsers', 'Manage Users') },
    { key: 'can_edit_users', label: t('roleManagement.editUsers', 'Edit Users') },
  ];

  useEffect(() => {
    setLoading(true);
    fetch('/api/postgres/roles')
      .then(res => res.json())
      .then(data => setRoles(data.roles || []))
      .catch(() => setRoles([]))
      .finally(() => setLoading(false));
  }, []);

  const handlePermissionChange = (roleIdx: number, permKey: string, value: boolean) => {
    setRoles(prev => prev.map((role, idx) => idx === roleIdx ? {
      ...role,
      permissions: { ...role.permissions, [permKey]: value }
    } : role));
  };

  const handleSave = async (role: any) => {
    setLoading(true);
    setError(null);
    setSuccess(null);
    try {
      const res = await fetch('/api/roles/update-permissions', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          role_id: role.role_id,
          permissions: role.permissions,
          current_user_role_id: 0 // Only superuser can use this UI
        })
      });
      if (!res.ok) throw new Error('Failed to update permissions');
      setSuccess('Permissions updated!');
    } catch (err: any) {
      setError(err.message || 'Failed to update permissions');
    } finally {
      setLoading(false);
    }
  };

  if (!userPermissions?.can_manage_users) return null;

  if (isMobile) {
    return (
      <div className="space-y-4 mt-6 pb-6 sm:pb-6" style={{ paddingBottom: '96px' }}>
        <h2 className={`text-xl font-bold px-4 ${isRTL ? 'text-right' : ''}`}>{t('roleManagement.title', 'Role Management')}</h2>
        {error && <div className="text-red-500 mb-2 px-4">{t('roleManagement.error', error)}</div>}
        {success && <div className="text-green-600 mb-2 px-4">{t('roleManagement.success', success)}</div>}
        {roles.map((role, idx) => (
          <Card key={role.role_id} className="w-[90vw] mx-auto">
            <CardContent className="py-4 space-y-2">
              <div className="flex justify-between">
                <span className="font-medium">{t('roleManagement.roleName', 'Role Name')}</span>
                <span>{role.role_name}</span>
              </div>
              {PERMISSIONS.map(perm => (
                <div key={perm.key} className="flex justify-between items-center">
                  <span>{perm.label}</span>
                  <input
                    type="checkbox"
                    checked={!!role.permissions?.[perm.key]}
                    onChange={e => handlePermissionChange(idx, perm.key, e.target.checked)}
                    disabled={role.role_id === 0}
                  />
                </div>
              ))}
              <Button
                size="sm"
                onClick={() => handleSave(role)}
                disabled={loading || role.role_id === 0}
                style={{ background: '#0B674E', color: '#fff' }}
                className="w-full box-border mt-2"
              >
                {t('roleManagement.save', 'Save')}
              </Button>
            </CardContent>
          </Card>
        ))}
      </div>
    );
  }

  return (
    <Card className="mt-6">
      <CardHeader>
        <CardTitle className={isRTL ? 'text-right' : ''}>{t('roleManagement.title', 'Role Management')}</CardTitle>
      </CardHeader>
      <CardContent className="px-1 sm:px-4">
        {error && <div className="text-red-500 mb-2">{t('roleManagement.error', error)}</div>}
        {success && <div className="text-green-600 mb-2">{t('roleManagement.success', success)}</div>}
        <div className="overflow-x-auto w-full">
          <table className="w-full min-w-[600px] text-sm border-collapse">
            <thead>
              <tr>
                <th className="p-2 border whitespace-nowrap">{t('roleManagement.roleName', 'Role Name')}</th>
                {PERMISSIONS.map(perm => (
                  <th key={perm.key} className="p-2 border whitespace-nowrap">{perm.label}</th>
                ))}
                <th className="p-2 border whitespace-nowrap">{t('roleManagement.actions', 'Actions')}</th>
              </tr>
            </thead>
            <tbody>
              {roles.map((role, idx) => (
                <tr key={role.role_id}>
                  <td className="p-2 border text-center">{role.role_name}</td>
                  {PERMISSIONS.map(perm => (
                    <td key={perm.key} className="p-2 border text-center">
                      <input
                        type="checkbox"
                        checked={!!role.permissions?.[perm.key]}
                        onChange={e => handlePermissionChange(idx, perm.key, e.target.checked)}
                        disabled={role.role_id === 0}
                        className={isMobile ? 'w-full box-border' : ''}
                      />
                    </td>
                  ))}
                  <td className="p-2 border text-center">
                    <Button
                      size="sm"
                      onClick={() => handleSave(role)}
                      disabled={loading || role.role_id === 0}
                      className={isMobile ? 'w-full box-border' : ''}
                    >{t('roleManagement.save', 'Save')}</Button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </CardContent>
    </Card>
  );
};

export default RoleManagement; 