import React, { useEffect, useState } from 'react';
import { Card, CardHeader, CardTitle, CardContent } from '../components/ui/card';
import { Button } from '../components/ui/button';
import { useAuth } from '../hooks/use-auth';
import { loadMqttLibrary, connectAndSend } from '../services/mqttService';
import { useTranslation } from 'react-i18next';

// Add valve commands
const VALVE_COMMANDS = {
  "OPEN_VALVE": "Jh8ARQ==",
  "CLOSE_VALVE": "Jh8BRg=="
};

function useIsMobile(breakpoint = 768) {
  const [isMobile, setIsMobile] = useState(false);
  useEffect(() => {
    const check = () => setIsMobile(window.matchMedia(`(max-width: ${breakpoint}px)`).matches);
    check();
    window.addEventListener('resize', check);
    return () => window.removeEventListener('resize', check);
  }, [breakpoint]);
  return isMobile;
}

const PendingApprovals: React.FC = () => {
  const { user, userPermissions } = useAuth();
  const [requests, setRequests] = useState<any[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState<string | null>(null);
  const [mqttLoaded, setMqttLoaded] = useState(false);
  const { t, i18n } = useTranslation();
  const isRTL = i18n.dir() === 'rtl';
  const isMobile = useIsMobile();

  useEffect(() => {
    if (!userPermissions?.can_manage_users) return;
    setLoading(true);
    fetch('/api/postgres/pending-max-value-requests')
      .then(res => res.json())
      .then(data => setRequests(data.requests || []))
      .catch(() => setRequests([]))
      .finally(() => setLoading(false));
  }, [userPermissions]);

  // Load MQTT library
  useEffect(() => {
    loadMqttLibrary()
      .then(() => {
        setMqttLoaded(true);
      })
      .catch((error) => {
        console.error("Failed to load MQTT library:", error);
      });
  }, []);

  const handleAction = async (id: number, action: 'approve' | 'reject') => {
    setLoading(true);
    setError(null);
    setSuccess(null);
    try {
      const res = await fetch('/api/postgres/pending-max-value-requests', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ id, action, reviewer: user?.email }),
      });
      if (!res.ok) throw new Error('Failed to update request');
      
      if (action === 'approve') {
        // Get the approved request details to check valve status
        const request = requests.find(r => r.id === id);
        if (request) {
          // Fetch current device data to check valve status
          try {
            const deviceDataRes = await fetch('/api/postgres/device-data', {
              method: 'POST',
              headers: { 'Content-Type': 'application/json' },
              body: JSON.stringify({ deviceName: request.device_id }),
            });
            
            if (deviceDataRes.ok) {
              const deviceData = await deviceDataRes.json();
              const currentValveState = deviceData.valve_state?.toLowerCase().trim();
              
              // If valve is closed and user has permission to open valve, automatically open it
              if (currentValveState === 'close valve' && userPermissions?.can_open_valve && mqttLoaded) {
                console.log('Valve is currently closed. Automatically sending OPEN_VALVE command after approval...');
                try {
                  // Use the actual dev_eui from the device data
                  const devEUI = deviceData.dev_eui || `8254812310000${request.device_id.slice(-3)}`;
                  await connectAndSend(devEUI, VALVE_COMMANDS["OPEN_VALVE"]);
                  setSuccess('Request approved and valve opened automatically!');
                } catch (valveError) {
                  console.error('Failed to automatically open valve after approval:', valveError);
                  setSuccess('Request approved! Note: Failed to automatically open valve.');
                }
              } else {
                setSuccess('Request approved!');
              }
            } else {
              setSuccess('Request approved!');
            }
          } catch (deviceDataError) {
            console.error('Failed to fetch device data for valve check:', deviceDataError);
            setSuccess('Request approved!');
          }
        } else {
          setSuccess('Request approved!');
        }
      } else {
        setSuccess('Request rejected!');
      }
      
      setRequests(reqs => reqs.filter(r => r.id !== id));
    } catch (err: any) {
      setError(err.message || 'Failed to update request');
    } finally {
      setLoading(false);
    }
  };

  if (!userPermissions?.can_manage_users) return null;

  // Мобильный режим: карточки
  if (isMobile) {
    return (
      <div className="space-y-4 mt-6">
        <h2 className={`text-xl font-bold px-4 ${isRTL ? 'text-right' : ''}`}>{t('pendingApprovals.title', 'Pending Max Value Change Approvals')}</h2>
        {error && <div className="text-red-500 mb-2 px-4">{t('pendingApprovals.error', error)}</div>}
        {success && <div className="text-green-600 mb-2 px-4">{t('pendingApprovals.success', success)}</div>}
        {requests.length === 0 && <div className="text-center text-muted-foreground px-4">{t('pendingApprovals.noRequests', 'No pending requests')}</div>}
        {requests.map((req) => (
          <Card key={req.id} className="w-full">
            <CardContent className="py-4 space-y-2">
              <div className="flex justify-between">
                <span className="font-medium">{t('pendingApprovals.deviceId', 'Device ID')}</span>
                <span>{req.device_id}</span>
              </div>
              <div className="flex justify-between">
                <span className="font-medium">{t('pendingApprovals.requestedValue', 'Requested Value')}</span>
                <span>{req.requested_value}</span>
              </div>
              <div className="flex justify-between">
                <span className="font-medium">{t('pendingApprovals.requestedBy', 'Requested By')}</span>
                <span>{req.requested_by}</span>
              </div>
              <div className="flex justify-between">
                <span className="font-medium">{t('pendingApprovals.requestedAt', 'Requested At')}</span>
                <span>{new Date(req.requested_at).toLocaleString()}</span>
              </div>
              <div className="flex gap-2 pt-2">
                <Button size="sm" onClick={() => handleAction(req.id, 'approve')} disabled={loading} style={{ background: '#0B674E', color: '#fff' }} className="flex-1">
                  {t('pendingApprovals.approve', 'Approve')}
                </Button>
                <Button size="sm" onClick={() => handleAction(req.id, 'reject')} disabled={loading} style={{ background: '#D74322', color: '#fff' }} className="flex-1">
                  {t('pendingApprovals.reject', 'Reject')}
                </Button>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>
    );
  }

  return (
    <Card className="mt-6">
      <CardHeader>
        <CardTitle className={isRTL ? 'text-right' : ''}>{t('pendingApprovals.title', 'Pending Max Value Change Approvals')}</CardTitle>
      </CardHeader>
      <CardContent className="px-1 sm:px-4">
        {error && <div className="text-red-500 mb-2">{t('pendingApprovals.error', error)}</div>}
        {success && <div className="text-green-600 mb-2">{t('pendingApprovals.success', success)}</div>}
        <div className="overflow-x-auto w-full">
          <table className="w-full min-w-[600px] text-sm border-collapse">
            <thead>
              <tr>
                <th className="p-2 border whitespace-nowrap">{t('pendingApprovals.deviceId', 'Device ID')}</th>
                <th className="p-2 border whitespace-nowrap">{t('pendingApprovals.requestedValue', 'Requested Value')}</th>
                <th className="p-2 border whitespace-nowrap">{t('pendingApprovals.requestedBy', 'Requested By')}</th>
                <th className="p-2 border whitespace-nowrap">{t('pendingApprovals.requestedAt', 'Requested At')}</th>
                <th className="p-2 border whitespace-nowrap">{t('pendingApprovals.actions', 'Actions')}</th>
              </tr>
            </thead>
            <tbody>
              {requests.map((req) => (
                <tr key={req.id}>
                  <td className="p-2 border text-center">{req.device_id}</td>
                  <td className="p-2 border text-center">{req.requested_value}</td>
                  <td className="p-2 border text-center">{req.requested_by}</td>
                  <td className="p-2 border text-center">{new Date(req.requested_at).toLocaleString()}</td>
                  <td className={`p-2 border`}> 
                    <div className="flex gap-2 justify-center items-center w-full">
                      <Button size="sm" onClick={() => handleAction(req.id, 'approve')} disabled={loading} style={{ background: '#0B674E', color: '#fff' }} className="w-full box-border">
                        {t('pendingApprovals.approve', 'Approve')}
                      </Button>
                      <Button size="sm" onClick={() => handleAction(req.id, 'reject')} disabled={loading} style={{ background: '#D74322', color: '#fff' }} className="w-full box-border">
                        {t('pendingApprovals.reject', 'Reject')}
                      </Button>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </CardContent>
    </Card>
  );
};

export default PendingApprovals; 