import { NextResponse } from 'next/server';
import { postgresClient } from '../../../../lib/postgresql-client';

export async function POST(req: Request) {
  try {
    const body = await req.json();
    const { action, deviceId, flags, newMaxValue, userRoleId, userPermissions } = body;

    switch (action) {
      case 'initialize_threshold_columns':
        // Idempotent operation: Add columns if they don't exist
        // Note: Column names with % need to be double-quoted in SQL.
        await postgresClient.query(`
          DO $$
          BEGIN
            IF NOT EXISTS (SELECT 1 FROM information_schema.columns 
                           WHERE table_name = 'device_max_value' AND column_name = '80%') THEN
              ALTER TABLE device_max_value 
              ADD COLUMN "80%" BOOLEAN DEFAULT false,
              ADD COLUMN "85%" BOOLEAN DEFAULT false,
              ADD COLUMN "90%" BOOLEAN DEFAULT false,
              ADD COLUMN "100%" BOOLEAN DEFAULT false;
            END IF;
          END $$;
        `);
        return NextResponse.json({ message: 'Database columns initialized/verified.' });

      case 'get_command_flags':
        if (!deviceId) return NextResponse.json({ error: 'Device ID is required' }, { status: 400 });
        const flagsResult = await postgresClient.query(
          'SELECT "80%", "85%", "90%", "100%" FROM device_max_value WHERE device_name = $1',
          [deviceId]
        );
        if (flagsResult.rows.length > 0) {
          return NextResponse.json({ flags: flagsResult.rows[0] });
        } else {
          // If no record for device, return default false flags
          return NextResponse.json({ flags: { "80%": false, "85%": false, "90%": false, "100%": false } });
        }

      case 'update_command_flags':
        if (!deviceId || !flags) return NextResponse.json({ error: 'Device ID and flags are required' }, { status: 400 });
        await postgresClient.query(
          'UPDATE device_max_value SET "80%" = $1, "85%" = $2, "90%" = $3, "100%" = $4 WHERE device_name = $5',
          [flags["80%"], flags["85%"], flags["90%"], flags["100%"], deviceId]
        );
        return NextResponse.json({ message: 'Command flags updated.' });

      case 'update_max_value':
        // Only allow direct update if user is superuser (role_id=0, can_approve_maximum)
        if (!userPermissions?.can_approve_maximum) {
          // Instead of returning 403, create a pending approval request
          try {
            await postgresClient.query(
              `INSERT INTO pending_max_value_requests 
               (device_id, requested_value, requested_by, requested_at, status)
               VALUES ($1, $2, $3, NOW(), 'pending')`,
              [deviceId, newMaxValue, userPermissions?.email || 'Unknown']
            );
            return NextResponse.json({ 
              message: 'Approval request created successfully',
              requiresApproval: true 
            });
          } catch (error: any) {
            console.error('Failed to create approval request:', error);
            return NextResponse.json({ 
              error: 'Failed to create approval request',
              details: error.message 
            }, { status: 500 });
          }
        }

        console.log('[update_max_value] Received payload:', { deviceId, newMaxValue, type_deviceId: typeof deviceId, type_newMaxValue: typeof newMaxValue });
        // Coerce newMaxValue to a number if it's a string
        const parsedMaxValue = typeof newMaxValue === 'string' ? Number(newMaxValue) : newMaxValue;
        if (!deviceId || typeof parsedMaxValue !== 'number' || isNaN(parsedMaxValue)) {
          console.log('[update_max_value] 400 error: missing or invalid deviceId/newMaxValue', { deviceId, newMaxValue });
          return NextResponse.json({ error: 'Device ID and new max value are required' }, { status: 400 });
        }
        // Ensure device exists in device_max_value
        const insertResult = await postgresClient.query(
          `INSERT INTO device_max_value (device_name, max_value)
           VALUES ($1, $2)
           ON CONFLICT (device_name) DO NOTHING`,
          [deviceId, parsedMaxValue]
        );
        console.log('[update_max_value] Insert result:', insertResult.rowCount);
        // Now update as before
        const updateResult = await postgresClient.query(
          'UPDATE device_max_value SET max_value = $1, "80%" = false, "85%" = false, "90%" = false, "100%" = false WHERE device_name = $2 RETURNING max_value',
          [parsedMaxValue, deviceId]
        );
        console.log('[update_max_value] Update result:', updateResult.rows);
        if (updateResult.rows.length > 0) {
          return NextResponse.json({
            message: 'Max value updated and flags reset.',
            newMaxValue: updateResult.rows[0].max_value,
            resetFlags: { "80%": false, "85%": false, "90%": false, "100%": false }
          });
        } else {
          console.log('[update_max_value] 404 error: device not found or no change', { deviceId, newMaxValue });
          return NextResponse.json({ error: 'Failed to update max value, device not found or no change.' }, { status: 404 });
        }

      default:
        return NextResponse.json({ error: 'Invalid action' }, { status: 400 });
    }
  } catch (error: any) {
    console.error('[DEVICE_MANAGEMENT_API_ERROR]', error);
    return NextResponse.json({ error: 'Internal Server Error', details: error.message }, { status: 500 });
  }
} 