"use client";

import React, { useEffect, useMemo, useState } from "react";
import { Skeleton } from '../components/ui/skeleton';
import { Alert, AlertDescription, AlertTitle } from '../components/ui/alert';
import { AlertCircle } from "lucide-react";
import { loadMqttLibrary, connectAndSend, MQTT_COMMANDS } from '../services/mqttService';
import { useTranslation } from 'react-i18next';

interface GaugeChartProps {
  deviceName: string;  // This is the primary identifier, matches device_name in database
  devEUI: string;      // This is the dev_eui from devices table
  unit?: string; // 'm³' or 'kWh'
  onValveStatusChange?: (status: { icon: string; label: string } | null) => void;
}

const GaugeChart: React.FC<GaugeChartProps> = ({
  deviceName,
  devEUI,
  unit = 'm³',
  onValveStatusChange
}) => {
  const { t } = useTranslation();
  const [svgContent, setSvgContent] = useState<string>('');
  const [mqttStatus, setMqttStatus] = useState<string | null>(null);
  const [mqttLoaded, setMqttLoaded] = useState(false);
  const [deviceData, setDeviceData] = useState<{
    current_read: number;
    max_value: number;
    valve_state: string;
    battery_voltage: number;
    device_type?: string;
  } | null>(null);
  const [loading, setLoading] = useState(true);

  // Load MQTT library on component mount
  useEffect(() => {
    loadMqttLibrary()
      .then(() => {
        setMqttLoaded(true);
      })
      .catch((error) => {
        console.error("Failed to load MQTT library:", error);
        setMqttStatus("Failed to load MQTT library");
      });
  }, []);

  // Fetch device data from PostgreSQL
  useEffect(() => {
    const fetchDeviceData = async () => {
      setLoading(true);
      try {
        //console.log('Fetching data for device:', deviceName);
        const response = await fetch('/api/postgres/device-data', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            deviceName: deviceName
          })
        });

        //console.log('API Response status:', response.status);
        const responseText = await response.text();
        //console.log('API Response text:', responseText);

        if (!response.ok) {
          throw new Error(`Failed to fetch device data: ${responseText}`);
        }

        let data;
        try {
          data = JSON.parse(responseText);
        } catch (e) {
          console.error('Failed to parse API response:', e);
          throw new Error('Invalid JSON response from API');
        }

        //console.log('Parsed device data:', data);
        
        if (!data || typeof data !== 'object') {
          console.error('Invalid data format:', data);
          throw new Error('Invalid data format received from API');
        }

        // Check if we got an empty object
        if (Object.keys(data).length === 0) {
          console.error('Empty data object received from API');
          throw new Error('No device data found for the specified device');
        }

        // Validate the data structure
        const requiredFields = {
          current_read: 'number',
          max_value: 'number',
          valve_state: 'string',
          battery_voltage: 'number'
        };

        const missingFields = Object.entries(requiredFields)
          .filter(([field, type]) => typeof data[field] !== type)
          .map(([field]) => field);

        if (missingFields.length > 0) {
          console.error('Missing or invalid fields:', {
            received: data,
            missingFields,
            expectedTypes: requiredFields
          });
          throw new Error(`Invalid data structure: missing or invalid fields: ${missingFields.join(', ')}`);
        }

        setDeviceData(data);
      } catch (error) {
        console.error('Error fetching device data:', error);
        setMqttStatus(error instanceof Error ? error.message : 'Failed to fetch device data');
        // Don't set deviceData to null here, keep the previous value if it exists
      } finally {
        setLoading(false);
      }
    };

    if (!deviceName) {
      console.warn('No deviceName provided to GaugeChart');
      return;
    }

    //console.log('Initial fetch for device:', deviceName);
    fetchDeviceData();
    // Set up polling every 600 seconds (10 minutes)
    const interval = setInterval(fetchDeviceData, 600000);
    return () => clearInterval(interval);
  }, [deviceName]);

  const polarToCartesian = (centerX: number, centerY: number, radius: number, angleInDegrees: number) => {
    const angleInRadians = (angleInDegrees - 90) * Math.PI / 180.0;
    return {
      x: centerX + radius * Math.cos(angleInRadians),
      y: centerY + radius * Math.sin(angleInRadians)
    };
  };

  const describeArc = (x: number, y: number, radius: number, startAngle: number, endAngle: number): string => {
    const start = polarToCartesian(x, y, radius, endAngle);
    const end = polarToCartesian(x, y, radius, startAngle);
    const largeArcFlag = endAngle - startAngle <= 180 ? "0" : "1";
    return `M${start.x} ${start.y} A${radius} ${radius} 0 ${largeArcFlag} 0 ${end.x} ${end.y}`;
  };

  const getArcColor = (percentage: number): string => {
    if (percentage >= 90) return "#F44336";
    if (percentage >= 75) return "#FF9800";
    if (percentage >= 50) return "#FFEB3B";
    if (percentage >= 25) return "#8BC34A";
    return "#4CAF50";
  };

  // Get valve status display info
  const getValveStatusInfo = () => {
    if (!deviceData) return null;
    
    const state = (deviceData.valve_state || '').trim().toLowerCase();
    const isWaterMeter = unit === 'm³';
    
    if (state === 'open valve') {
      return {
        icon: '/img/open.png',
        label: isWaterMeter ? t('gaugeChart.open', 'Open') : t('gaugeChart.on', 'On')
      };
    } else if (state === 'close valve') {
      return {
        icon: '/img/close.png',
        label: isWaterMeter ? t('gaugeChart.close', 'Close') : t('gaugeChart.off', 'Off')
      };
    }
    
    return null;
  };

  // Notify parent component about valve status changes
  useEffect(() => {
    if (!onValveStatusChange) return;
    
    const currentStatus = getValveStatusInfo();
    onValveStatusChange(currentStatus);
  }, [deviceData?.valve_state, unit]);

  useEffect(() => {
    // Use PostgreSQL data if available
    if (!deviceData) {
      //console.log('No device data available');
      setSvgContent('');
      return;
    }

    const currentReading = deviceData.current_read;
    const maxReading = deviceData.max_value;

    if (loading || currentReading === null || currentReading === undefined || 
        maxReading === null || maxReading === undefined || maxReading <= 0) {
      console.log('Skipping gauge render:', {
        loading,
        hasDeviceData: !!deviceData,
        currentRead: currentReading,
        maxValue: maxReading
      });
      setSvgContent('');
      return;
    }

    //console.log('Rendering gauge with data:', { currentReading, maxReading, deviceData });
    const percentage = (currentReading / maxReading) * 100;
    const clampedPercentage = Math.max(0, Math.min(percentage, 100));
    const color = getArcColor(clampedPercentage);

    const scale = 0.75;
    const centerX = 100;
    const centerY = 100;
    const radius = 80 * scale;
    const labelRadius = radius + 25;

    const arc = describeArc(centerX, centerY, radius, -100, -100 + (clampedPercentage / 100) * 200);
    const baseArc = describeArc(centerX, centerY, radius, -100, 100);

    const labelAngles = [-100, -50, 0, 50, 100];
    const labelValues = [0, 25, 50, 75, 100].map(p => ((p / 100) * maxReading).toFixed(1));
    const ticks = labelAngles.map((angle, i) => {
      const pos = polarToCartesian(centerX, centerY, labelRadius, angle);
      return `<text x="${pos.x}" y="${pos.y + 4}" text-anchor="middle" font-size="10" fill="#666">${labelValues[i]}</text>`;
    }).join('');

    setSvgContent(`
      <div style=\"display: flex; justify-content: center; width: 100%\">
        <svg viewBox=\"0 0 200 160\" style=\"width: 100%; max-width: 300px; height: auto; min-width: 160px;\" class=\"sm:max-w-full\">
          <path d=\"${baseArc}\" stroke=\"#ccc\" stroke-width=\"12\" fill=\"none\"/>
          <path d=\"${arc}\" stroke=\"${color}\" stroke-width=\"12\" fill=\"none\"/>
          ${ticks}
          <text x="${centerX}" y="${centerY - 10}" text-anchor="middle" font-size="16" fill="#333">${currentReading.toFixed(2)}</text>
          <text x="${centerX}" y="${centerY + 10}" text-anchor="middle" font-size="10" fill="#666">${unit}</text>
        </svg>
      </div>
    `);
  }, [deviceData, loading]);

  // Расширенный маппинг типа устройства к ключу перевода
  const DEVICE_TYPE_MAP: Record<string, string> = {
    'waterMeter': 'devices.waterMeter',
    'electricMeter': 'devices.electricMeter',
    'Water meter': 'devices.waterMeter',
    'Electric meter': 'devices.electricMeter',
    'Water Meter': 'devices.waterMeter',
    'Electric Meter': 'devices.electricMeter',
    // Добавьте другие типы устройств по мере необходимости
  };

  const getDeviceTypeLabel = (type: string) => {
    if (!type) return '';
    const key = DEVICE_TYPE_MAP[type.trim()] || type.trim();
    return t(key, type);
  };

  return (
    <div className="space-y-1 w-full max-w-md mx-auto">
      {loading ? (
        <Skeleton className="w-full h-[135px]" />
      ) : !deviceData ? (
        <Alert>
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>{t('gaugeChart.noData', 'No Data')}</AlertTitle>
          <AlertDescription>{t('gaugeChart.noDeviceData', 'No device data available')}</AlertDescription>
        </Alert>
      ) : (
        <>
          <div dangerouslySetInnerHTML={{ __html: svgContent }} className="w-full" />
          <div className="text-sm text-muted-foreground text-center">
            {unit === 'kWh' ? (
              <>
                {(() => {
                  const state = (deviceData.valve_state || '').trim().toLowerCase();
                  let status = deviceData.valve_state;
                  if (state === 'open valve') status = t('dashboard.online', 'Online');
                  else if (state === 'close valve') status = t('dashboard.offline', 'Offline');
                  return <>{t('dashboard.meterStatus', 'Meter status')}: {status} <br /></>;
                })()}
              </>
            ) : (
              <>
                {t('dashboard.battery', 'Battery')}: {deviceData.battery_voltage.toFixed(2)}V <br />
              </>
            )}
            {t('dashboard.maxAllowed', 'Max Allowed')}: {deviceData.max_value.toFixed(1)}{unit} <br />
            {t('dashboard.leftForUsage', 'Left for usage')}: {(deviceData.max_value - deviceData.current_read).toFixed(1)}{unit}
          </div>
        </>
      )}
      {mqttStatus && (
        <Alert variant="destructive">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>{t('common.error', 'Error')}</AlertTitle>
          <AlertDescription>{mqttStatus}</AlertDescription>
        </Alert>
      )}
    </div>
  );
};

export default GaugeChart;
